<?php

namespace App\Http\Controllers;

use App\Models\CartItem;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CartController extends Controller
{
    public function index()
    {
        $cartItems = CartItem::with(['product', 'inventoryItem'])->where('user_id', Auth::id())->get();

        $subtotal = $cartItems->sum(function ($item) {
            return $item->quantity * $item->price_at_add;
        });

        return view('cart.index', compact('cartItems', 'subtotal'));
    }

    public function store(Request $request, Product $product)
    {
        $request->validate([
            'quantity' => 'required|integer|min:1',
        ]);

        $inventoryItem = $product->inventoryItem;

        if (!$inventoryItem || $inventoryItem->available_quantity < $request->quantity) {
            return redirect()->route('cart.index')->with('error', 'Insufficient stock available.');
        }

        if ($request->quantity < $inventoryItem->minimum_order_quantity) {
            return redirect()->route('cart.index')->with('error', 'Quantity is below the minimum order requirement.');
        }

        $cartItem = CartItem::firstOrNew([
            'user_id' => Auth::id(),
            'product_id' => $product->id,
        ]);

        $cartItem->quantity += $request->quantity;
        $cartItem->price_at_add = $inventoryItem->current_price;
        $cartItem->save();

        return redirect()->route('cart.index')->with('status', 'Product added to cart.');
    }

    public function update(Request $request, CartItem $cartItem)
    {
        $this->authorize('update', $cartItem);

        $request->validate([
            'quantity' => 'required|integer|min:1',
        ]);

        $cartItem->update(['quantity' => $request->quantity]);

        return redirect()->route('cart.index')->with('status', 'Cart updated successfully.');
    }

    public function destroy(CartItem $cartItem)
    {
        $this->authorize('delete', $cartItem);

        $cartItem->delete();

        return redirect()->route('cart.index')->with('status', 'Item removed from cart.');
    }
}